<?php
namespace TrxAddons\AiHelper\Elementor;

use TrxAddons\AiHelper\OpenAi;
use TrxAddons\AiHelper\OpenAiAssistants;
use TrxAddons\AiHelper\Lists;
use TrxAddons\AiHelper\Utils;

if ( ! class_exists( 'Helper' ) ) {

    /**
	 * Main class for AI Helper Elementor
	 */
	class Helper {

		/**
		 * Constructor
		 */
		function __construct() {
			// Enqueue scripts and styles for the elementor
			add_action( 'elementor/editor/after_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
			add_action( 'elementor/editor/after_enqueue_styles', array( $this, 'enqueue_styles' ) );
			add_filter( 'trx_addons_filter_localize_script_admin', array( $this, 'localize_script' ) );

			// AJAX callback for the 'Elementor Generate text' buttons
			add_action( 'wp_ajax_trx_addons_ai_helper_elementor_generate_text', array( $this, 'elementor_generate_text' ) );
			// Callback function to fetch answer from the assistant
			add_action( 'wp_ajax_trx_addons_ai_helper_elementor_generate_text_fetch', array( $this, 'fetch_answer' ) );
		}

        /**
		 * Check if AI Helper is allowed
		 */
		public static function is_allowed() {
			// return OpenAi::instance()->get_api_key() != ''
			// 		|| OpenAiAssistants::instance()->get_api_key() != '';
            return true;
		}

		/**
		 * Enqueue scripts for the elementor
		 * 
		 * @hooked 'elementor/editor/after_enqueue_scripts'
		 */
		function enqueue_scripts() {
			if ( self::is_allowed() ) {
				wp_enqueue_script( 'trx_addons-ai-helper-elementor-generate-text', trx_addons_get_file_url( TRX_ADDONS_PLUGIN_ADDONS . 'ai-helper/support/Elementor/assets/js/index.js' ), array( 'jquery' ), null, true );
			}
		}

		/**
		 * Enqueue styles for the elementor
		 * 
		 * @hooked 'elementor/editor/after_enqueue_styles'
		 */
		function enqueue_styles() {
			if ( self::is_allowed() ) {
				wp_enqueue_style( 'trx_addons-ai-helper-elementor-generate-text', trx_addons_get_file_url( TRX_ADDONS_PLUGIN_ADDONS . 'ai-helper/support/Elementor/assets/css/index.css' ) );
			}
		}

		/**
		 * Localize script to show messages
		 * 
		 * @hooked 'trx_addons_filter_localize_script_admin'
		 * 
		 * @param array $vars  Array of variables to be passed to the script
		 * 
		 * @return array  Modified array of variables
		 */
		function localize_script( $vars ) {
			if ( self::is_allowed() ) {

				$vars['elm_ai_generate_text_btn_label'] = esc_html__( "Generate texts with AI", 'trx_addons' );

				$vars['elm_ai_generate_text_modal_close'] = esc_html__( "Close", 'trx_addons' );

				$vars['elm_ai_generate_text_modal_submit'] = esc_html__( "Submit", 'trx_addons' );

				$vars['elm_ai_generate_text_modal_purpose_label'] = esc_html__( "Purpose of the block", 'trx_addons' );

				$vars['elm_ai_generate_text_modal_purpose_pl'] = esc_html__( "About Us", 'trx_addons' );

			}
			return $vars;
		}

		/**
		 * Send a query to API to process text
		 * 
		 * @hooked 'elementor_generate_text'
		 * 
		 * @param WP_REST_Request  $request  Full details about the request.
		 */
		function elementor_generate_text() {

			trx_addons_verify_nonce();

			$answer = array(
				'error' => '',
				'data' => array(
					'text' => '',
					'message' => ''
				)
			);

			$helper = trx_addons_get_option( 'ai_helper_elementor_generate_text_model' );
			$description = trx_addons_get_option( 'ai_helper_elementor_generate_text_description' );
			$content_post = json_decode( trx_addons_get_value_gp( 'content' ), true );

			if ( ! $content_post ) {

				$answer['error'] = $content_post;

				trx_addons_ajax_response( $answer );

			}

			$content_prime = $this->parse_element( $content_post );

			if ( rand(0, 1) === 1 ) {
				$answer['data']['content'] = $content_prime;
			} else {
				$answer['finish_reason'] = 'queued';
				$answer['fetch_time'] = 500;
				$answer['thread_id'] = 1;
				$answer['run_id'] = 1;
			}

			// Return response to the AJAX handler
			trx_addons_ajax_response( $answer );
		}

		/**
		 * Callback function to fetch answer from the assistant
		 * 
		 * @hooked 'wp_ajax_trx_addons_ai_helper_elementor_generate_text_fetch'
		 */
		function fetch_answer() {

			trx_addons_verify_nonce();

			$run_id = trx_addons_get_value_gp( 'run_id' );
			$thread_id = trx_addons_get_value_gp( 'thread_id' );

			$answer = array(
				'error' => '',
				'finish_reason' => 'queued',
				'run_id' => $run_id,
				'thread_id' => $thread_id,
				'data' => array(
					'text' => '',
					'message' => ''
				)
			);

			$this->elementor_generate_text();
		}

		/**
		 * parse_element
		 *
		 * @param  mixed $element
		 * @return void
		 */
		function parse_element( $element ) {

			if ( ( 'container' == $element['elType'] || 'section' == $element['elType'] || 'column' == $element['elType'] ) && ! empty( $element['elements'] ) ) {

				$time_elements = array();
				foreach( $element['elements'] as $sub_element ) {

					$time_elements[] = $this->parse_element( $sub_element );
				}

				$element['elements'] = $time_elements;

			} elseif ( 'widget' == $element['elType'] ) {

				$element = $this->update_widget( $element );

			}

			return $element;
		}

		/**
		 * update_widget
		 *
		 * @param  mixed $widget
		 * @return void
		 */
		function update_widget( $widget ) {

			if ( 'heading' == $widget['widgetType'] ) {

				$widget['settings']['title'] = '111' . $widget['settings']['title'];

			} elseif ( 'text-editor' == $widget['widgetType'] ) {

				$widget['settings']['editor'] = '111' . $widget['settings']['editor'];

			}

			return $widget;
		}
    }
}